#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.
# generic driver

from fenrirscreenreader.core import debug
from fenrirscreenreader.core.speechDriver import speech_driver


class driver(speech_driver):
    """Debug speech driver for Fenrir development and testing.
    
    This driver provides speech output to console/debug output instead of
    actual audio, making it useful for development, testing, and debugging
    scenarios where audio output is not needed or available.
    
    All speech operations are logged to console with descriptive messages,
    allowing developers to trace speech behavior without audio hardware.
    """
    def __init__(self):
        speech_driver.__init__(self)

    def initialize(self, environment):
        """Initialize the debug speech driver.
        
        Args:
            environment: Fenrir environment dictionary
        """
        self._is_initialized = True
        self.env = environment
        print("Speech Debug Driver: Initialized")

    def shutdown(self):
        """Shutdown the debug speech driver."""
        if self._is_initialized:
            self.cancel()
        self._is_initialized = False
        print("Speech Debug Driver: Shutdown")

    def speak(self, text, queueable=True, ignore_punctuation=False):
        """Output speech text to console for debugging.
        
        Args:
            text (str): Text to speak
            queueable (bool): Whether speech can be queued
            ignore_punctuation (bool): Whether to ignore punctuation
        """
        if not self._is_initialized:
            return
        if not queueable:
            self.cancel()
        print("Speech Debug Driver: Speak:" + text)
        print("Speech Debug Driver: -----------------------------------")

    def cancel(self):
        """Log speech cancellation to console."""
        if not self._is_initialized:
            return
        print("Speech Debug Driver: Cancel")

    def set_callback(self, callback):
        """Log callback setting to console.
        
        Args:
            callback: Callback function (logged but not used)
        """
        print("Speech Debug Driver: set_callback")

    def clear_buffer(self):
        """Log buffer clearing to console."""
        if not self._is_initialized:
            return
        print("Speech Debug Driver: clear_buffer")

    def set_voice(self, voice):
        """Log voice setting to console.
        
        Args:
            voice: Voice setting (logged but not used)
        """
        if not self._is_initialized:
            return
        print("Speech Debug Driver: set_voice:" + str(voice))

    def set_pitch(self, pitch):
        """Log pitch setting to console.
        
        Args:
            pitch: Pitch setting (logged but not used)
        """
        if not self._is_initialized:
            return
        print("Speech Debug Driver: set_pitch:" + str(pitch))

    def set_rate(self, rate):
        """Log rate setting to console.
        
        Args:
            rate: Rate setting (logged but not used)
        """
        if not self._is_initialized:
            return
        print("Speech Debug Driver: set_rate:" + str(rate))

    def set_module(self, module):
        """Log module setting to console.
        
        Args:
            module: Module setting (logged but not used)
        """
        if not self._is_initialized:
            return
        print("Speech Debug Driver: set_module:" + str(module))

    def set_language(self, language):
        """Log language setting to console.
        
        Args:
            language: Language setting (logged but not used)
        """
        if not self._is_initialized:
            return
        print("Speech Debug Driver: set_language:" + str(language))

    def set_volume(self, volume):
        """Log volume setting to console.
        
        Args:
            volume: Volume setting (logged but not used)
        """
        if not self._is_initialized:
            return
        print("Speech Debug Driver: set_volume:" + str(volume))
