---
name: Membrane
category: Solid Mechanics
layout: example
---

## Compute the deflection of a 2d clamped membrane
The geometry is an ellipse, clamped on ${\Gamma_1}$  at altitude $z$ and  pulled down by a volumic force f.
~~~freefem
func z = x;
// Mesh
real theta = 4.*pi/3.;
real a = 2., b = 1.; // Half the length of the major and minor axis
border Gamma1(t=0, theta)    {x=a*cos(t); y=b*sin(t);}
border Gamma2(t=theta, 2*pi) {x=a*cos(t); y=b*sin(t);}
mesh Th = buildmesh(Gamma1(100) + Gamma2(50));
// Fespace
fespace Vh(Th,P2); // P2 conforming triangular finite element space
Vh phi, w, f = 1;
~~~
Note that $z$ is a reserved word (used for 3D applications) but it can be overriden.

The equations for the membrane are
$$
-\Delta \phi=f, ~~\phi|_{\Gamma_1}=z,
$$
And in variational form,
$$
  \int_{\Omega}\nabla \phi \cdot\nabla w = \int_\Omega f w,\quad  ~\phi|_{\Gamma_1}=z,
$$
for all $w\in V$, where $V(\Omega)=\{ w\in (H(\Omega)^2): w|_{\Gamma_1}=0\}$.
~~~freefem
solve Laplace(phi, w)
       =int2d(Th)(  dx(phi)*dx(w) + dy(phi)*dy(w))
       - int2d(Th)(f*w) + on(Gamma1, phi=z);

// Plot
plot(Th,phi, wait=true, fill=true,dim=3,ps="membrane.ps");
~~~
The graphics in FreeFem are minimal.

![][_solution]

Publication quality graphics can be generated by calling Paraview or gnuplot.
Here is an example of export to gnupot
~~~freefem
{
  ofstream ff("graph.txt");
  for (int i = 0; i < Th.nt; i++) {
    for (int j = 0; j < 3; j++)
      ff << Th[i][j].x  << "    " << Th[i][j].y << "  " << phi[][Vh(i,j)] << endl;
  ff << Th[i][0].x << "    " << Th[i][0].y << "  " << phi[][Vh(i,0)] << endl << endl << endl;
  }
}
// Save the mesh. Could be reused in another script with readmesh
savemesh(Th, "Th.msh");
~~~
Then call gnuplot in a terminal window and type
set palette rgbformulae 30,31,32
splot "graph.txt" w l pal
![][_gnuplot]

[_solution]: https://raw.githubusercontent.com/FreeFem/FreeFem-markdown-figures/main/examples/examples/membrane/solution.png

[_gnuplot]: https://raw.githubusercontent.com/FreeFem/FreeFem-markdown-figures/main/examples/examples/membrane/gnuplot.png
