/*************************************************************************
* Copyright (C) 2022 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     Digesting message according to SHA256
//
//  Contents:
//        ippsHashStateMethodSet_SHA256_NI()
//
*/

#include "owndefs.h"
#include "owncp.h"
#include "hash/pcphash.h"
#include "hash/pcphash_rmf.h"
#include "pcptool.h"
#include "hash/sha256/pcpsha256stuff.h"

/*F*
//    Name: ippsHashStateMethodSet_SHA256_NI
//
// Purpose: Setup SHA256 method inside the hash state
// (using the Intel® Secure Hash Algorithm - New Instructions (Intel® SHA-NI) instruction set).
//
// Returns:                Reason:
//    ippStsNullPtrErr           pMethod == NULL or pState == NULL
//    ippStsNotSupportedModeErr  mode disabled by configuration
//    ippStsNoErr                no errors
//
*F*/
/* clang-format off */
IPPFUN(IppStatus, ippsHashStateMethodSet_SHA256_NI, (IppsHashState_rmf* pState,
                                                     IppsHashMethod* pMethod))
/* clang-format on */
{
    /* test pointers */
    IPP_BAD_PTR2_RET(pState, pMethod);

    HASH_METHOD(pState) = pMethod;

#if (_SHA_NI_ENABLING_ == _FEATURE_TICKTOCK_ || _SHA_NI_ENABLING_ == _FEATURE_ON_)
    pMethod->hashAlgId     = ippHashAlg_SHA256;
    pMethod->hashLen       = IPP_SHA256_DIGEST_BITSIZE / 8;
    pMethod->msgBlkSize    = MBS_SHA256;
    pMethod->msgLenRepSize = MLR_SHA256;
    pMethod->stateLen      = IPP_SHA256_STATE_BYTESIZE;
    pMethod->hashInit      = sha256_hashInit;
    pMethod->hashUpdate    = sha256_ni_hashUpdate;
    pMethod->hashOctStr    = sha256_hashOctString;
    pMethod->msgLenRep     = sha256_msgRep;

    return ippStsNoErr;
#else
    pMethod->hashAlgId     = ippHashAlg_Unknown;
    pMethod->hashLen       = 0;
    pMethod->msgBlkSize    = 0;
    pMethod->msgLenRepSize = 0;
    pMethod->stateLen      = 0;
    pMethod->hashInit      = NULL;
    pMethod->hashUpdate    = NULL;
    pMethod->hashOctStr    = NULL;
    pMethod->msgLenRep     = NULL;

    return ippStsNotSupportedModeErr;
#endif
}
