# -*- coding: utf-8 -*-
#
# Copyright 2012-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Qt specific tests for the main module."""

# pylint: disable=F0401
from PyQt4 import QtCore
from twisted.internet import defer

from ubuntu_kylin_sso.main import qt
from ubuntu_kylin_sso.tests import TestCase


class MiscTestCase(TestCase):
    """Tests for module level misc functions."""

    def clean_timers(self):
        """Remove all timers."""
        for timer in qt.TIMERS:
            timer.stop()
        qt.TIMERS = set()

    @defer.inlineCallbacks
    def setUp(self):
        yield super(MiscTestCase, self).setUp()
        self.addCleanup(self.clean_timers)

    def test_timeout_func_ignores_priority(self):
        """Should not pass priority as argument."""
        qt.timeout_func(0, self._set_called, priority="crazy")
        QtCore.QCoreApplication.instance().processEvents()
        self.assertEqual(self._called, ((), {}))

    def test_timeout_func_keeps_reference(self):
        """Should keep a reference to the timer created."""
        self.assertEqual(len(qt.TIMERS), 0)
        qt.timeout_func(0, self._set_called, priority="crazy")
        self.assertEqual(len(qt.TIMERS), 1)

    def test_timeout_func_removes_reference(self):
        """Remove reference to timer when callback returns False."""
        self.assertEqual(len(qt.TIMERS), 0)
        qt.timeout_func(0, self._set_called, priority="crazy")
        self.assertEqual(len(qt.TIMERS), 1)
        QtCore.QCoreApplication.instance().processEvents()
        self.assertEqual(len(qt.TIMERS), 0)

    def test_timeout_func_timer_activation(self):
        """The timer should be activated with the right period."""
        self.assertEqual(len(qt.TIMERS), 0)
        qt.timeout_func(1337, self._set_called, priority="crazy")
        self.assertEqual(len(qt.TIMERS), 1)
        self.assertEqual(list(qt.TIMERS)[0].interval(), 1337)

    def test_timeout_func_argument_passing(self):
        """The right arguments should be passed to the callback."""
        qt.timeout_func(0, self._set_called, 1, name="a")
        QtCore.QCoreApplication.instance().processEvents()
        self.assertEqual(self._called, ((1,), {"name": "a"}))

    def test_start_setup_calls_turkish_fix(self):
        """The start_setup function calls the turkish fix function."""
        self.patch(qt.QtCore, "QCoreApplication",
                   lambda *args, **kwargs: object())
        self.patch(qt.dbus.mainloop.qt, "DBusQtMainLoop",
                   lambda **kwargs: None)
        called = []
        self.patch(qt, "fix_turkish_locale", lambda: called.append(1))

        qt.start_setup()
        self.assertEqual(called, [1])
